/* Copyright (C) 2012-2018 RealVNC Ltd. All Rights Reserved.
 * 
 * This is sample code intended to demonstrate part of the
 * VNC Mobile Solution SDK. It is not intended as a production-ready
 * component.
 */

#ifndef UUID_4f440a5d_8c54_7ad5_3e27_e20c267bfa15
#define UUID_4f440a5d_8c54_7ad5_3e27_e20c267bfa15

#include <VNCDeviceProviderBase.h>
#include "UsbDevice.h"

#ifdef _MSC_VER
#include "UsbDeviceInfoProviderWin.h"
#elif defined(VNC_QNX)
#include "UsbDeviceInfoProviderQnx.h"
#else
#include "UsbDeviceInfoProviderUnix.h"
#endif /* _MSC_VER */

namespace vncdeviceprovider_usb
{
/**
 * Interface for an observer listening for new/removed USB devices.
 *
 * \see UsbDeviceInfoProvider
 */
class UsbDeviceInfoObserver
{
public:
  virtual ~UsbDeviceInfoObserver();

  /**
   * Provides a device that has just been added. This can only be called
   * while the info provider is started (by calling startMonitoring()).
   */
  virtual void usbDeviceAdded(const UsbDevice& dev) = 0;

  /**
   * Information for a device that has been removed. This can only be called
   * while the info provider is started (by calling startMonitoring()).
   */
  virtual void usbDeviceRemoved(vnc_uint32_t id) = 0;

  /**
   * The observer must (asynchonously) call back the
   * UsbDeviceInfoProvider::handleTimeout method after the interval has
   * elapsed.
   *
   * \param usecs The interval in microseconds.
   */
  virtual void setTimerForUsbDeviceInfoProvider(
              VNCDiscoverySDKTimeoutMicroseconds usecs) = 0;
};

/**
 * Platform independent wrapper class for monitoring and enumerating the USB
 * devices. Owns and uses a platform specific implementation.
 *
 * \see UsbDevice
 */
class UsbDeviceInfoProvider
{
public:
  /**
   * Constructor.
   *
   * \param deviceProvider The DeviceProvider class.
   * \param observer The observer that will receive the call backs with
   * new/removed devices and handle the timer for the UsbDeviceInfoProvider.
   */
  inline UsbDeviceInfoProvider(VNCDeviceProviderBase& deviceProvider, 
                        UsbDeviceInfoObserver& observer)
    : mImpl(deviceProvider, observer)
  {
  }

  /**
   * Destructor. All the devices returned by this class should have been
   * deleted before this destructor is invoked.
   */
  inline ~UsbDeviceInfoProvider()
  {
  }

  /**
   * Starts the provider monitoring for new/removed devices.
   *
   * Uses the filter filter to decide on the type of devices being monitored.
   * By default there is no filter.
   *
   * Filters are defined in the DeviceFilter enumeration.
   *
   * Once started, the user of the class has to retrieve the event handle and
   * monitor for events on it. When events occur, it must call handleEvent().
   *
   * Calling startMonitoring while monitoring has not effect and the function
   * will return true.
   *
   * \return true If monitoring can be started.
   */
  inline bool startMonitoring()
  {
    return mImpl.startMonitoring();
  }

  /**
   * Stops monitoring for new/removed devices.
   */
  inline void stopMonitoring()
  {
    mImpl.stopMonitoring();
  }

  /**
   * Starts the enumeration of the device. Call nextDevice() to get the next
   * available device.
   *
   * Uses the filter filter to decide on the type of devices being monitored.
   * By default there is no filter.
   *
   * Filters are defined in the DeviceFilter enumeration.
   *
   * Once started, the user of the class has to retrieve the event handle and
   * monitor for events on it. When events occur, it must call handleEvent().
   *
   * Calling start enumeration, while another enumeration is ongoing causes the
   * first one to be stopped (re-starts enumeration).
   *
   * \return true If the filter can be added.
   */
  inline bool startEnumeration()
  {
    return mImpl.startEnumeration();
  }

  /**
   * Stops the enumeration of the device.
   */
  inline void stopEnumeration()
  {
    mImpl.stopEnumeration();
  }

  /**
   * Gets the next device from the enumeration. Before this is called,
   * startEnumeration() should have been called.
   *
   * \return A new device object that represents the device. When no longer
   * needed, it should be deleted. (Ownership is passed to the caller).
   */
  inline UsbDevice* nextDevice()
  {
    return mImpl.nextDevice();
  }

  /**
   * Handles an event that occured on the event handle.
   */
  inline void handleEvent(VNCDiscovererEventHandle* pHandle)
  {
    mImpl.handleEvent(pHandle);
  }

  /**
   * Handles a timeout.
   */
  inline void handleTimeout()
  {
    mImpl.handleTimeout();
  }

  /**
   * Sets a property on the info provider.
   *
   * Supported properites:
   *  - VNCUsbDeviceProviderPropertyDhcpScript - The location of the dhcpscript to
   *  run when a new network interface is detected. The script will receive as
   *  argument the name of the network interface.
   */
  inline VNCDiscoverySDKError setProperty(const char *pProperty, 
                            const char *pValue)
  {
    return mImpl.setProperty(pProperty, pValue);
  }

  /**
   * Gets the value of a property.
   *
   * \return The value of the property, or NULL if the property is not
   * supported.
   */
  inline const char* getProperty(const char *pProperty)
  {
    return mImpl.getProperty(pProperty);
  }

  /**
   * Gets the even handle (should be retrieved after startMonitoring()).
   */
  inline size_t getEventHandles(VNCDiscovererEventHandle *& handles)
  {
    return mImpl.getEventHandles(handles);
  }

  /**
   * \brief Gets the name of the device assigned for the interface.
   *
   * This is the name of the device as is found in /dev on Linux systems. At
   * the moment Windows and QNX implementations don't return anything.
   *
   * \param pSysPath The system path to the device.
   * \param configurationValue The configuration value for which to retrieve
   * the information.
   * \param interfaceNumber The number of the interface for which to retrieve
   * the information.
   * \param altenrateSetting The alternate setting of the interface for which
   * to retrieve the information.
   * \param interfaceType The type of interface. At the moment it's either tty,
   * or hidraw.
   * \param pDevName Pointer to the buffer to fill with the device name. E.g.:
   * the name will be ttyACM1 for '/dev/ttyACM1'.
   * \param devNameSize The size of the buffer (the maximum size of the name
   * will be this - 1, to allow for the terminating NUL character).
   *
   * \return The size of the name. If no name is found, then this is 0 and the
   * buffer is left as it was. If the buffer is not big enough this will be the
   * required size, and the buffer will be left as it was.
   */
  inline size_t getDevName(const char* pSysPath, vnc_uint8_t configurationValue,
      vnc_uint8_t interfaceNumber, vnc_uint8_t alternateSetting,
      UsbDevice::InterfaceType interfaceType,
      char* pDevName, size_t devNameSize)
  {
    return mImpl.getDevName(pSysPath, configurationValue, interfaceNumber,
        alternateSetting, interfaceType, pDevName, devNameSize);
  }

private:
#ifdef _MSC_VER
  UsbDeviceInfoProviderWin mImpl;
#elif defined(VNC_QNX)
  UsbDeviceInfoProviderQnx mImpl;
#else
  UsbDeviceInfoProviderUnix mImpl;
#endif /* _MSC_VER */
};

} // namespace vncdeviceprovider_usb

#endif /* !defined (UUID_4f440a5d_8c54_7ad5_3e27_e20c267bfa15) */

